<?php

namespace App\Models;

use DB;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Member extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'reg_no',
        'profile_picture',
        'email',
        'proof',
        'age',
        'gender',
        'address',
        'phone',
        'height',
        'weight',
        'type',
        'valid_from',
        'valid_to'
    ];

    /**
     * Define a has one relationship with the Membership model, retrieving the latest membership.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function latest_membership()
    {
        return $this->hasOne(Membership::class)->latestOfMany()->withDefault();
    }

    /**
     * Define a has many relationship with the Membership model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function memberships()
    {
        return $this->hasMany(Membership::class);
    }
 
    /**
     * Define a has many relationship with the Attendance model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function attendances()
    {
        return $this->hasMany(Attendance::class);
    }

    /**
     * Scope a query to include attendances for a specific month and year.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query  The query builder instance.
     * @param  int  $year                                     The year for which to include attendances.
     * @param  int  $month                                    The month for which to include attendances.
     * @return \Illuminate\Database\Eloquent\Builder          The modified query builder instance.
     */
    public function scopeWithAttendanceInMonth($query, $year, $month)
    {
        return $query->with([
            'attendances' => function ($query) use ($year, $month) {
                $query->whereYear('attendance_date', $year)
                    ->whereMonth('attendance_date', $month);
            }
        ]);
    }

    public function scopeWithAttendance($query, $date)
    {
        return $query->with([
            'attendances' => function ($query) use ($date) {
                $query->where('attendance_date', $date);
            }
        ]);
    }
}